// ---------------------------------------------------------------------------
//
//  This file is part of PermLib.
//
// Copyright (c) 2009-2011 Thomas Rehn <thomas@carmen76.de>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// ---------------------------------------------------------------------------


#ifndef LINEAR_FORM_LIST_H_
#define LINEAR_FORM_LIST_H_

#include <boost/cstdint.hpp>
#include <boost/dynamic_bitset.hpp>

namespace permlib {

/// a linear form with 0,1-coefficients
typedef boost::dynamic_bitset<> LinearForm;

/// action of a permutation on a linear form
template<class PERM>
struct LinearFormAction {
	/// action
	LinearForm operator()(const PERM &p, const LinearForm& set) const {
		LinearForm ret(set.size());
		for (unsigned int i = 0; i < ret.size(); ++i) {
			if (set[i])
				ret.set(p / i, 1);
		}
		return ret;
	}
};


/// a list of linear forms
/**
 * For invariants generated by Dade's algorithm we store them as product of linear forms.
 * The factors make up the list.
 */
class LinearFormList {
public:
	/// evaluates this invariant at a given (bit)set
	/**
	 * Data type of Z must support 64-bit multiplication. 
	 * If moduloPrime is set additionally modulo operation is required.
	 *
	 * @param l the set to evaluate
	 * @param moduloPrime if set, result is evaluated modulo 0xFFFFFFFBUL, the largest 32-bit prime
	 * @return returns the value of the invariant at $l or 1 if invariant is empty
	 */
	template<typename Z>
	Z evaluate(const LinearForm& l, bool moduloPrime = true) const;
	
	/// adds a linear form to the list
	void add(const LinearForm& l) { m_forms.push_back(l); }
	/// number of elements in the list (equals the polynomial degree)
	unsigned long size() const { return m_forms.size(); }
private:
	std::list<LinearForm> m_forms;
};

template<typename Z>
Z LinearFormList::evaluate(const LinearForm& l, bool moduloPrime) const {
	Z result(1);
	BOOST_FOREACH(const LinearForm& form, m_forms) {
		BOOST_ASSERT(form.size() == l.size());
		
		LinearForm merge(form);
		merge &= l;
		const Z value(merge.count());
		if (value == 0) {
			return 0;
		}
		result *= value;
		if (moduloPrime)
			result %= 0xFFFFFFFBUL;
	}
	return result;
}

}

#endif // LINEAR_FORM_LIST_H_
